C      *****************************************************************
C      * Subroutine FluxKT                                             *
C      * Controls the calculation of new known totals of components due*
C      * to water flux. The water fluxes are put through an error      *
C      * routine to ensure that they do not violate mass balance       *
C      * restrictions.  Next, the layer from which the water is moving *
C      * is determined; then the amount of each component in soluble   *
C      * form within that layer (as a free component and soluble       *
C      * dependent species) is calculated.  A fraction                 *
C      * (flux/soil water) of this soluble total is moved to the       *
C      * appropriate layer.  All components in transit are calculated, *
C      * for each layer, before any movement takes place.  If the      *
C      * movement were calculated first, the concentration in a        *
C      * subsequent layer would be changed, whereas we want the        *
C      * calculations to be done on the concentrations from the last   *
C      * time step.                                                    *
C      *                                                               *
C      * Variables:                                                    *
C      * CatchArea -Input - A vector, of size NLayers, of the catchment*
C      *                    area for each layer.  Input from TREGRO, or*
C      *                    read into the model by the routine InParam.*
C      *                    Units are m**2.                            *
C      *                    (Common block VModel, file VModel.f)       *
C      * ElapT     -Input - The elapsed time since the last call, i.e.,*
C      *                    the model timestep.  (This is model time,  *
C      *                    not real time.)  Input from TREGRO, or read*
C      *                    into the model by the routine InParam.     *
C      *                    Units are s.                               *
C      *                    (Common block VModel, file VModel.f)       *
C      * F         -Local - Flux index.  Unitless.                     *
C      * GoAhead   -Output- Indicates whether any errors were found.   *
C      *                    (GoAhead=1, no errors found; GoAhead=0,    *
C      *                    errors found.)  Unitless.                  *
C      * Liters    -Local - The water flux in units of liters.  Units  *
C      *                    are L.                                     *
C      * LNum      -Local - Layer index.  Unitless.                    *
C      * LPerM3    -Local - Constant for converting m**3 to liters.    *
C      *                    Units are L/m**3.                          *
C      * NComp     -Input - The number of components in the system.    *
C      *                    Unitless.                                  *
C      *                    (Common block VModel, file VModel.f)       *
C      * NLayers   -Input - The number of soil layers in the system.   *
C      *                    Input from TREGRO, or read into the model  *
C      *                    by the routine InParam.  Unitless.         *
C      *                    (Common block VPos, file VModel.f)         *
C      * NLayersSize                                                   *
C      *           -Input - The max number of layers, used to size     *
C      *                    arrays.  Unitless.                         *
C      * SoilWat   -Local - The soil water content.  Units are L.      *
C      * WFlux     -Input - A vector, of size NLayers, of net water    *
C      *                    fluxes between layers.  Flux(0) is the net *
C      *                    flux between the atmosphere and the top    *
C      *                    layer, Flux(1) is the net flux between     *
C      *                    layers number 1 and 2, Flux(3) is the net  *
C      *                    flux between layers number 3 and 4.  If    *
C      *                    the flux is positive, the water is moving  *
C      *                    down; if the flux is negative, the water is*
C      *                    moving up.  Input from TREGRO, or read into*
C      *                    the model by the routine InParam.  Units   *
C      *                    are L/(m**2 sec).                          *                     
C      *                    (Common block VModel, file VModel.f)       *
C      *****************************************************************
       SUBROUTINE FLUXKT(GOAHEAD)
							IMPLICIT NONE
							INCLUDE 'VMODEL.INC'
       INTEGER F, GOAHEAD, LNUM, LPERM3/1000/
							REAL*8 LITERS, SOILWAT
							
C      *------------------------------------------*
C      * Indicate that no errors have been found. *
C      *------------------------------------------*
       GOAHEAD = 1

C      *-------------------------------------------------------------*
C      * Check that fluxes do not violate mass balance restrictions. *
C      *-------------------------------------------------------------*
       CALL FLUXERR(GOAHEAD)
	      IF (GOAHEAD.NE.1) GOTO 9999

C      *-------------------------------------------------------------------*
C      * Calculate the nutrients moving into the first layer (calling when *
C      * flux = 0 will essentially zero out yesterday's values which is    *
C      * necessary!).  If water is evaporating from the first layer, no    *
C      * nutrients are lost.                                               *
C      *-------------------------------------------------------------------*
       IF (WFLUX(0).GE.0) THEN 
	         CALL FLOP
	      ENDIF
	   
C      *-------------------------------------------------*
C      * Loop over all fluxes that occur between layers. *
C      *-------------------------------------------------*
       DO 200 F = 1, NLAYERS

C         *----------------------------------------------------*
C         * Determine the layer that the ions are moving from. *
C         *----------------------------------------------------*
          IF (WFLUX(F).GE.0) THEN
	            LNUM = F
	         ELSE
	            LNUM = F+1
	         ENDIF

C         *-----------------------------------------*
C         * Calculate the soil water in this layer. *   
C         *-----------------------------------------*
          SOILWAT = CATCHAREA(LNUM) * THICKNESS(LNUM) * 
     >              OLDTHETA(LNUM) * LPERM3

C         *---------------------------------------------------------*
C         * Calculate the amount of soluble components that should  *
C         * be moved from this layer.  Note that the flux is passed *
C         * to FSolub in units of liters (same as SoilWat).         *
C         *---------------------------------------------------------*
          LITERS = WFLUX(F) * CATCHAREA(F) * ELAPT
          CALL FSOLUB(LNUM, LITERS, SOILWAT)
		  
  200  CONTINUE
  
C      *-----------------------------------------------------------------*
C      * Look at the movement of ions in the fluxes above and below each *
C      * layer and calculate new known totals for all components.  Check *
C      * that all new known totals are not less than zero.               *
C      *-----------------------------------------------------------------*
       CALL FMOVION(GOAHEAD)
	      IF (GOAHEAD.NE.1) GOTO 9999

C      *--------------------------------*
C      * Escape hatch for fatal errors. *
C      *--------------------------------*
 9999  CONTINUE	 
 
       RETURN
	      END
C      *****************************************************************
C      *                    END SUBROUTINE                             *
C      *****************************************************************
